// SPDX-License-Identifier: MIT
pragma solidity ^0.8.23; //4; //23;

import "inetworkv1.sol"; // holds all interfaces
import "istandard.sol"; // all standard Interfaces
import "library.sol"; // Math, signedMath and Strings
import "abs_ownable.sol";

import "abs_metadatav1.sol"; 

/*
The gallery functionality builds upon the IMetadataV1 functionality that allows the owner to set a path,
name and hash. When used as a gallery, the name is a JSON file that contains gallery information. 

When the high level code queries this contract, it will find that it supports IGalleryV2 where it can ask
if the gallery is ready to be queried. If so, it will call projectDataCurrent() to get the latest
version of the gallery.
*/

interface IGalleryV2 {
    // returns the current gallery index
    function IsGalleryReady() external returns(bool);
}

abstract contract ERC165 is IERC165 {
    function supportsInterface(bytes4 interfaceId) public view virtual returns (bool) {
        return interfaceId == type(IERC165).interfaceId;
    }
}

//
// Main class
//
contract GALV2 is  Ownable, ERC165, MetaDataV1, IPennyOracleV1, IGalleryV2  {
    using Strings for uint256;

    address private m_currentPennyOracle; // type IPennyOracleV1

    constructor(address _theTitled, address _thePennyOracle) 
        Ownable(_theTitled, msg.sender )
        {
            m_currentPennyOracle = _thePennyOracle;
        }


    //  ------------------  IERC165 interface ------------------ 
    /**
     * @dev See {IERC165-supportsInterface}.
     */
    function supportsInterface(bytes4 interfaceId) public view virtual override returns (bool) {
        return interfaceId == type(IPennyOracleV1).interfaceId ||
            interfaceId == type(IMetadataV1).interfaceId ||
            interfaceId == type(IGalleryV2).interfaceId ||
            super.supportsInterface(interfaceId);
    }

    function Interfaces() public pure returns(bytes4 theNumber) {
        return type(IGalleryV2).interfaceId;
    }

    // For checking to see if the gallery has been setup.

    function IsGalleryReady() public view returns(bool) {
        return projectActive();
    }


    //  ------------------  IMetadataV1 interface ------------------ 

    /**
     * @dev This routine sets the initial state for the project metadata. The _URI is used
     * as the base URI for all token URIs. The _Project name represents the file name(without
     * extension) on the server pointed to by _URI. The _Project JSON file can be hashed to
     * produce the SHA1 hash value represented by _hash. Only the owner of the contract can
     * set the project location. If the project moves, the latest addition will be considered
     * the working location.
     */
    function projectUpdate(string calldata _URI, string calldata _Project, string calldata _hash) public onlyEither {
        require(bytes(_URI).length > 0,"Project base URI needs valid path");
        require(bytes(_Project).length > 0,"Project name length invalid");
        require(bytes(_hash).length == 40,"requires SHA1 hash string");
        return iProjectUpdate(_URI,_Project,_hash);
    }


    //  ------------------  IPennyOracleV1 interface ------------------ 
    //
    // The Penny Oracle functionality
    //
    // This NFT uses the Penny Oracle. It is set in the constructor, but if it changes
    // we can update that source here.
    //
    function pennyOracleSet(address _PennyOracle) public onlyOwner {
        require(_PennyOracle != address(0), "No Penny to Reference");
        require(IERC165(_PennyOracle).supportsInterface(type(IPennyOracleV1).interfaceId),"doesn't support IPennyOracleV1");
        iPennyOracleSet(_PennyOracle);
    }

    function iPennyOracleSet(address _PennyOracle) internal {
        m_currentPennyOracle = _PennyOracle;
    }

    function pennyOracleStatus() public returns (bool bActive, address theContract, uint256 lastBlock) {
        require(m_currentPennyOracle != address(0), "No Penny to Reference");
        return IPennyOracleV1(m_currentPennyOracle).pennyOracleStatus();
    }

    function pennyPriceTfuel() public view returns (uint256 pennyTfuelWei, string memory priceOfTfuel) {
        require(m_currentPennyOracle != address(0), "No Penny to Reference");
        return IPennyOracleV1(m_currentPennyOracle).pennyPriceTfuel();
    }


}
