// SPDX-License-Identifier: MIT
pragma solidity ^0.8.23;


/**
 * @dev Provides information about the current execution context, including the
 * sender of the transaction and its data. While these are generally available
 * via msg.sender and msg.data, they should not be accessed in such a direct
 * manner, since when dealing with meta-transactions the account sending and
 * paying for execution may not be the actual sender (as far as an application
 * is concerned).
 *
 * This contract is only required for intermediate, library-like contracts.
 */
abstract contract Context {
    function _msgSender() internal view virtual returns (address) {
        return msg.sender;
    }

    function _msgData() internal view virtual returns (bytes calldata) {
        return msg.data;
    }

    function _contextSuffixLength() internal view virtual returns (uint256) {
        return 0;
    }
}

/**
 * @dev Contract module which provides a basic access control mechanism, where
 * there is an account (an owner) that can be granted exclusive access to
 * specific functions.
 *
 * The initial owner is set to the address provided by the deployer. This can
 * later be changed with {transferOwnership}.
 *
 * This module is used through inheritance. It will make available the modifier
 * `onlyOwner`, which can be applied to your functions to restrict their use to
 * the owner.
 */
abstract contract Ownable is Context {
    address private m_titledAddr; // the property owner for this project
    address private m_ownerAddr;  // the smart contract opperator

    /**
     * @dev The caller account is not authorized to perform an operation.
     */
    error OwnableUnauthorizedAccount(address account);

    /**
     * @dev The owner is not a valid owner account. (eg. `address(0)`)
     */
    error OwnableInvalidOwner(address owner);

    event OwnershipTransferred(address indexed previousOwner, address indexed newOwner);

    /**
     * @dev Initializes the contract setting the address provided by the deployer as the initial owner.
     */
    constructor(address _theTitledAddr, address _ownerAddr ) {
        if (_theTitledAddr == address(0)) {
            revert OwnableInvalidOwner(address(0));
        }
        if (_ownerAddr == address(0)) {
            revert OwnableInvalidOwner(address(0));
        }
        _transferTitle(_theTitledAddr);
        _transferOwnership(_ownerAddr);
    }

    /**
     * @dev Throws if called by any account other than the owner.
     */
    modifier onlyOwner() {
        _checkOwner();
        _;
    }

    /**
     * @dev Throws if called by any account other than the owner.
     */
    modifier onlyTitled() {
        _checkTitled();
        _;
    }

    /**
     * @dev Throws if called by any account other than ether The Titled or The Owner.
     */
    modifier onlyEither() {
        _checkEither();
        _;
    }

    /**
     * @dev Returns the address of the current owner.
     */
    function owner() public view virtual returns (address) {
        return m_ownerAddr;
    }
    /**
     * @dev Returns the address of the current titled.
     */
    function titled() public view virtual returns (address) {
        return m_titledAddr;
    }

    /**
     * @dev Throws if the sender is not the owner.
     */
    function _checkOwner() internal view virtual {
        if (owner() != _msgSender()) {
            revert OwnableUnauthorizedAccount(_msgSender());
        }
    }
    /**
     * @dev Throws if the sender is not the owner.
     */
    function _checkTitled() internal view virtual {
        if (titled() != _msgSender()) {
            revert OwnableUnauthorizedAccount(_msgSender());
        }
    }

    /**
     * @dev Throws if the sender is not either The Titled or The Owner.
     */
    function _checkEither() internal view virtual {
        if( titled() != _msgSender() ) {
            if( owner() != _msgSender() ) {
                revert OwnableUnauthorizedAccount(_msgSender());
            }
        }
    }

    /**
     * @dev Leaves the contract without owner. It will not be possible to call
     * `onlyOwner` functions. Can only be called by the current owner.
     *
     * NOTE: Renouncing ownership will leave the contract without an owner,
     * thereby disabling any functionality that is only available to the owner.
     * Only the Title address can setup a new owner.
     */
    function renounceOwnership() public virtual onlyEither {
        _transferOwnership(address(0));
    }

    /**
     * @dev Transfers ownership of the contract to a new account (`newOwner`).
     * Can only be called by the current owner.
     */
    function transferOwnership(address newOwner) public virtual onlyTitled {
        if (newOwner == address(0)) {
            revert OwnableInvalidOwner(address(0));
        }
        _transferOwnership(newOwner);
    }

    /**
     * @dev Transfers ownership of the contract to a new account (`newOwner`).
     * Internal function without access restriction.
     */
    function _transferOwnership(address newOwner) internal virtual {
        address oldOwner = m_ownerAddr;
        m_ownerAddr = newOwner;
        emit OwnershipTransferred(oldOwner, newOwner);
    }

    /**
     * @dev Transfers ownership of the contract to a new account (`newOwner`).
     * Internal function without access restriction.
     */
    function _transferTitle(address newTitle) internal virtual {
        address oldTitled = m_titledAddr;
        m_titledAddr = newTitle;
        emit OwnershipTransferred(oldTitled, newTitle);
    }
}

