// SPDX-License-Identifier: MIT
pragma solidity ^0.8.23;

import "inetworkv1.sol";
import "library.sol";
//
// impliments the base Metadata hash functionality
//
abstract contract MetaDataV1 is IMetadataV1 {
    using Strings for uint256;

    // for managing the project base location. The owner of the contract can push a 
    // new location with hash. The callers can request any of the recorded locations.
    struct Location {
        string sBase; // holds the URI for the base location like: "https://amorstyle.com/nfts/kalamazoo/"
        string sProject; // holds the main JSON file name like: "kalamazoo"
        string sHash; // holds the hash of the project
    }
    Location[] private m_Loc; 

    /**
     * @dev This routine sets the initial state for the project metadata. The _URI is used
     * as the base URI for all token URIs. The _Project name represents the file name(without
     * extension) on the server pointed to by _URI. The _Project JSON file can be hashed to
     * produce the SHA1 hash value represented by _hash. Only the owner of the contract can
     * set the project location. If the project moves, the latest addition will be considered
     * the working location.
     */
    function iProjectUpdate(string calldata _URI, string calldata _Project, string calldata _hash) internal {
        m_Loc.push(Location(_URI,_Project,_hash));
    }

    /**
     * @dev check to make sure things are setup.
     */
    function projectActive() internal view returns(bool) {
        if( m_Loc.length > 0) {
            return true;
        }
        return false;
    }

    /**
     * @dev Index to working project.
     */
    function projectIndex() public view returns(uint uiIndex) {
        require(m_Loc.length > 0,"not setup yet");
        return (m_Loc.length - 1);
    }
    /**
     * @dev project data by index.
     */
    function projectData(uint _uiIndex) public view returns(string memory URI, string memory Project,string memory hash) {
        require(_uiIndex < m_Loc.length,"index out of bounds");
        return (m_Loc[_uiIndex].sBase,m_Loc[_uiIndex].sProject,m_Loc[_uiIndex].sHash);
    }
    /**
     * @dev Most current project data.
     */
    function projectDataCurrent() public view returns(string memory URI, string memory Project,string memory hash) {
        return projectData(projectIndex());
    }
    /**
     * @dev Most current project base URI.
     */
    function projectBaseURI() public view returns(string memory URI) {
        return m_Loc[projectIndex()].sBase;
    }
}
