// SPDX-License-Identifier: MIT
pragma solidity ^0.8.23;

import "INetworkV1.sol";
import "Library.sol";
import "abs_Ownable.sol";

//
// This is the key class that manages who can cliam coin that this contract holds. This class works
// off the idea that there is a "The Titled" account which is the property owner. There is an 'owner'
// which is the account that opperates the contract. And then there are Agents and It. Agents are
// the 'sales people' and It is the technical infrastructure. 
//
// Note that when minting, the mint function will check that both an agent and It address are provided.
//
//
// dependent upon Ownable for title() and owner()
abstract contract PayV1 is IPayV1, Ownable, IPennyOracleV1 {

    // Every address is treated like an account
    struct acct {
        uint256 uiBlock;
        uint256 uiWei;
        uint256 uiCount;
    }
    mapping(address addr => acct) private m_accounts;

    // in order to be able to sweep accounts, we have to be able to find them. Each time
    // we add a new payout, we'll add it to our sweepable list.
    mapping(uint256 Id => address account) private m_accountsById;
    uint256 private m_totalAccounts;

    address private m_currentPennyOracle; // type IPennyOracleV1

    uint256 private m_treasuryBalance = 0;
    
    uint256 private constant ONE_HOUR = 600;
    uint256 private constant ONE_DAY = (24 * ONE_HOUR);
    uint256 private m_InactivityDeadline = 0;

    uint256 private m_mintPennies = 0; // set at launch and afterwards
    uint256 private m_minCount; 
    uint256 private m_cutPercentAgents; 
    uint256 private m_cutPercentWebsite; 

    bool private m_bActive = false;

    /**
     * @dev The constructor sets up basic ownership.
     * @param _theTitled The Titled property owner
     * @param _theOwner The smart contract opperator
     * @param _thePennyOracle Source fro tfuel price
     */

    // The constructor sets up TheTitled account
    constructor(address _theTitled, address _theOwner, address _thePennyOracle)
            Ownable(_theTitled,_theOwner) {
        m_currentPennyOracle = _thePennyOracle;
        m_totalAccounts = 0; // used for sweeping dead accounts
    }

    // Need to set the state variables
    function payInitalize(uint256 _mintPennies, uint256 _timeoutDays, uint256 _minCount, uint256 _cutPercentAgent, uint256 _cutPercentWebsite) internal {

        m_mintPennies = _mintPennies;
        m_InactivityDeadline = (_timeoutDays * ONE_DAY);
        m_minCount = _minCount;
        m_cutPercentAgents = _cutPercentAgent;
        m_cutPercentWebsite = _cutPercentWebsite;
        m_bActive = true;
    }

    //
    // When reporting the commissions, the cuts are relative to the cost of minting. Because the project
    // owner has the ability to adjust the mint price, each time that happens, the cuts need to be recalculated.
    // It is expected that the project owner will/may adjust the mint price every once in a while. Like on
    // the granularity of once a month.
    //
    function iPayMintPenniesSet(uint256 _mintPennies) internal {
        m_mintPennies = _mintPennies;
    }

    function iPayMintPenniesRead() internal view returns(uint256) {
        return m_mintPennies;
    }

    function iPayActive() internal view returns (bool) {
        return m_bActive; 
    }

    // if the pay functionality is not setup or disabled, this routine will return false. true otherwise.
    function payActive() public view returns (bool) {
        return iPayActive(); 
    }

    // ----------------- IPennyOracleV1 interface functions ------------------------
    //
    // The IPay interface uses the penny oracle. Because of this, we're going to export it's interface
    // so contracts that offer it's pricing can fetch common data.
    //

    function iPennyOracleSet(address _PennyOracle) internal {
        m_currentPennyOracle = _PennyOracle;
    }

    function pennyOracleStatus() public returns (bool bActive, address theContract, uint256 lastBlock) {
        require(m_currentPennyOracle != address(0), "No Penny to Reference");
        return IPennyOracleV1(m_currentPennyOracle).pennyOracleStatus();
    }

    function pennyPriceTfuel() public view returns (uint256 pennyTfuelWei, string memory priceOfTfuel) {
        require(m_currentPennyOracle != address(0), "No Penny to Reference");
        return IPennyOracleV1(m_currentPennyOracle).pennyPriceTfuel();
    }




    // ----------------- external IPayV1 interface functions ------------------------

    /**
     * @dev This routine returns the commission split to the caller.
     */
    function payMintInfo() external view returns (uint256 payAgentPennies, uint256 payWebsitePennies, uint256 inactivityDeadline, uint256 minMintCount) {

        require( iPayActive(),"Not setup yet, MintInfo");

        uint256 uiMintSegment;
        bool bSuccess;
        (bSuccess, uiMintSegment) =  Math.tryDiv(m_mintPennies, 100);
        require(bSuccess,"Division overflow");

        (bSuccess, payAgentPennies) = Math.tryMul(uiMintSegment,m_cutPercentAgents);
        require(bSuccess,"math issue");
        (bSuccess, payWebsitePennies) = Math.tryMul(uiMintSegment,m_cutPercentWebsite);
        require(bSuccess,"math issue");

        // TODO: Need to return m_mintPennies
        return (payAgentPennies, payWebsitePennies, m_InactivityDeadline, m_minCount);
    }

    /**
     * @dev This function returns the balance along with more.
     */
    function payAgentBalance(address _account) external view returns(uint256 payBalanceWei, uint256 payDeadline, uint256 payCount) {
        require( iPayActive(),"Not setup yet, AgentBalance");
        require(_account != address(0),"Invalid account");
        return (m_accounts[_account].uiWei, (m_accounts[_account].uiBlock + m_InactivityDeadline), m_accounts[_account].uiCount);
    }

    /**
     * @dev This function returns the balance of the treasury.
     */
    function payTreasuryBalance() external view returns(uint256 BalanceWei) {
        require( iPayActive(),"Not setup yet,TreasuryBalance");
        return m_treasuryBalance;
    }

    // ----------------- external IPayV1 interface functions ------------------------

    // routines that get called frequently...

    // returns:
    //  - the requiredTfuel for mint
    //  - a penny's worth of tfuel
    //  - a friendly dscriptor of Tfuel
    //
    function currentWei(uint256 _pennies) internal view returns(uint256 mintWei) {

        uint tfuelWei;
        (tfuelWei, ) = IPennyOracleV1(m_currentPennyOracle).pennyPriceTfuel();

        //bool bSuc;
        (/*bSuc*/, mintWei) = Math.tryMul(tfuelWei, _pennies);
        // require success here or throw?
        return mintWei;
    }

    // Validates that the amount sent is the current mint amount
    function payValidateAmount() internal returns(bool bSuccess) {
        bSuccess = false;
        if( currentWei(m_mintPennies) == msg.value ) {
            bSuccess = true;
        }
        return bSuccess;
    }

    function payMintPrice() external view returns(uint256 mintWei) {
        require(m_bActive==true,"pay Not setup yet, MintPrice");
        return currentWei(m_mintPennies);
    }




    function iPayMintInfoWei() internal view returns (uint256 payFoundersWei, uint256 payPartnersWei, uint256 payWebsiteWei) {

        uint256 mintWei = currentWei(m_mintPennies);

        uint256 uiSegmentWei;
        bool bSuccess;
        (bSuccess, uiSegmentWei) =  Math.tryDiv(mintWei, 100);
        require(bSuccess,"Division overflow");

        uint256 payAgentsWei;
        (bSuccess, payAgentsWei) = Math.tryMul(uiSegmentWei,m_cutPercentAgents);
        require(bSuccess,"math issue");

        (bSuccess, payWebsiteWei) = Math.tryMul(uiSegmentWei,m_cutPercentWebsite);
        require(bSuccess,"math issue");

        // Now break the agent into 90/10
        (bSuccess, uiSegmentWei) =  Math.tryDiv(payAgentsWei, 100);
        require(bSuccess,"Division overflow");
        (bSuccess, payFoundersWei) = Math.tryMul(uiSegmentWei,10);
        require(bSuccess,"math issue");
        (bSuccess, payPartnersWei) = Math.tryMul(uiSegmentWei,90);
        require(bSuccess,"math issue");

        return (payFoundersWei,payPartnersWei,payWebsiteWei);
    }

    function updateAccount(address _anAccount, uint256 _valueWei) internal {

        acct storage theAcct = m_accounts[_anAccount];
        // if this is the first time we've see this account, theAcct.uiCount will be zero. If that is the case,
        // we will add this account to the sweepable mapping
        if( 0 == theAcct.uiCount ) {
            m_totalAccounts++;  // make space for a new account
            m_accountsById[m_totalAccounts]=_anAccount;
        }

        // if the account is new, the initial values are zero, thus all we need to 
        // do is add to those values.
        theAcct.uiBlock = block.number; // new timer starts
        theAcct.uiCount += 1;       // new commission count
        theAcct.uiWei += _valueWei;  // add valueWei.

        //if( m_accountsById)
    }
    function updateAbandonedAccount(address _anAccount) internal returns (bool bSuccess) {
        
        uint256 uiWei = m_accounts[_anAccount].uiWei; // grab the value
        m_accounts[_anAccount].uiWei = 0;    // clear out account
        m_accounts[_anAccount].uiCount = 0;  //

        // The Titled gets a new block number and the coin.
        m_accounts[titled()].uiBlock = block.number; // new timer starts
        m_accounts[titled()].uiWei = uiWei;  // claim valueWei.
        // The overall count of coin in the treasury does not change. 
        return true;
    }

    // for spreading the coin to the different accounts.
    function mintSplit(address _founder, address _partner, address _website) internal {
        uint256 newCoinWei = msg.value;

        uint256 uiFounderWei;
        uint256 uiPartnerWei;
        uint256 uiWebsiteWei;
        (uiFounderWei, uiPartnerWei, uiWebsiteWei) = iPayMintInfoWei();
        // TODO: need to make sure this can never happen and remove this throw!
        require((uiFounderWei + uiPartnerWei + uiWebsiteWei) < newCoinWei,"Out of Balance 1");

        // Now, split out the founder from the partner

        // Agent wei is split into founder(10%) and partner(90%)
        updateAccount(_founder, uiFounderWei);
        updateAccount(_partner, uiPartnerWei);
        updateAccount(_website, uiWebsiteWei);
        updateAccount(titled(), (newCoinWei - (uiFounderWei + uiPartnerWei + uiWebsiteWei)));

        m_treasuryBalance += newCoinWei;
    }

    // This routine is used by a contract to deliver payable funds to the founder account
    function titledPayment() internal {
        uint256 newCoinWei = msg.value;
        updateAccount(titled(), newCoinWei);
        m_treasuryBalance += newCoinWei;
    }

    // These conditions are used for anyone wanting to claim their balance from the treasury.
    function claimConditions(address _anAccount) internal view returns (bool bSuccess) {
        // Note that The Titled is always able to draw out their funds
        if( _anAccount != titled() ) {
            // Perform the two checks: the count is 5 or greater and did they delay?
            require(m_accounts[_anAccount].uiCount >= 5, "Sales count too low");
            require((m_accounts[_anAccount].uiBlock + m_InactivityDeadline) < block.number, "Waited too long");
        }
        return true;
    }
    // must call claimConditions before this. 
    function claimAccount(address _anAccount) internal returns(uint256 theBlanace) {

        uint256 theBalance = m_accounts[_anAccount].uiWei;
        m_accounts[_anAccount].uiWei = 0;
        m_treasuryBalance -= theBalance; // adjust the overall counter.
        return theBalance;
    }

    // funds are only sweepable if the count is below 5 and the account is inactive. 
    function sweepConditions(address _anAccount) internal view returns(bool bSuccess) {

        require(m_accounts[_anAccount].uiBlock > 0,"invalid account");
        require(m_accounts[_anAccount].uiWei > 0,"no balance");
        require( (m_accounts[_anAccount].uiBlock + m_InactivityDeadline) < block.number, "time not up yet");
        return true;
    }

    function IsSweepable(uint256 _id) internal view returns (address) {
        address anAccount = m_accountsById[_id];
        if( anAccount != address(0) ) {
            // only worth sweeping if there is a balance.
            if( m_accounts[anAccount].uiWei > 0 ) {
                // not allowed to sweep accounts that have not timed out. So, if the last transaction
                // on the block is longer than the inactivity deadline, we have something that
                // can be claimed.
                if( (m_accounts[anAccount].uiBlock + m_InactivityDeadline) < block.number ) {
                    return anAccount;
                }
            }

        }
        return address(0);
    }


    // Get a feeling for the number of accounts that need to be swept.
    function payTotalAccounts() public view returns(uint256 total) {
        total = m_totalAccounts;
    }
    //
    // In order to claim abbandoned funds, the delinquent accounts need to be found. This routine
    // will loop over the accounts and return when there is a claimable account.
    //
    // returns: 
    // - next start index (used for next call)
    // - a claimable address or address(0)
    //
    // Note that this account should not be called until at least the delay days have passed from mint. 
    // calling any earlier will just burn gas. Because the payTreasuryBalance() routine will return how
    // much coin is still held, the titled can get feel for whether to go looking or not. 
    //
    function sweepableId(uint256 _startIndex) public view onlyEither returns(address sweepableAddr, uint256 nextId) {
        // this routine loops over the collection of addresses looking to see if any meet the 
        // sweep conditions. If so, that id is returned to the caller and the index to the next
        // searchable account. 

        address Addr = address(0);
        if( _startIndex < m_totalAccounts ) {

            uint j=0; // limit loop 
            uint i;
            for (i = _startIndex; i <= m_totalAccounts; i++) {
                Addr = IsSweepable(i);
                if( Addr != address(0) ) {
                    // found an account that can be swept
                    nextId = i++;
                    return (Addr,nextId);
                }
                j++;
                if( j >= 10 ) {
                    break;
                }
            }
            return (address(0),i);
        } 
        return (address(0),m_totalAccounts);
    }
}
