// SPDX-License-Identifier: MIT
pragma solidity ^0.8.23;

import "INetworkV1.sol"; // interfaces.
import "IStandard.sol";  // standard interfaces



/**
 * @dev Provides information about the current execution context, including the
 * sender of the transaction and its data. While these are generally available
 * via msg.sender and msg.data, they should not be accessed in such a direct
 * manner, since when dealing with meta-transactions the account sending and
 * paying for execution may not be the actual sender (as far as an application
 * is concerned).
 *
 * This contract is only required for intermediate, library-like contracts.
 */
abstract contract Context {
    function _msgSender() internal view virtual returns (address) {
        return msg.sender;
    }

    function _msgData() internal view virtual returns (bytes calldata) {
        return msg.data;
    }

    function _contextSuffixLength() internal view virtual returns (uint256) {
        return 0;
    }
}

/**
 * @dev Contract module which provides a basic access control mechanism, where
 * there is an account (an owner) that can be granted exclusive access to
 * specific functions.
 *
 * The initial owner is set to the address provided by the deployer. This can
 * later be changed with {transferOwnership}.
 *
 * This module is used through inheritance. It will make available the modifier
 * `onlyOwner`, which can be applied to your functions to restrict their use to
 * the owner.
 */
abstract contract Ownable is Context {
    address private _owner;

    /**
     * @dev The caller account is not authorized to perform an operation.
     */
    error OwnableUnauthorizedAccount(address account);

    /**
     * @dev The owner is not a valid owner account. (eg. `address(0)`)
     */
    error OwnableInvalidOwner(address owner);

    event OwnershipTransferred(address indexed previousOwner, address indexed newOwner);

    /**
     * @dev Initializes the contract setting the address provided by the deployer as the initial owner.
     */
    constructor(address initialOwner) {
        if (initialOwner == address(0)) {
            revert OwnableInvalidOwner(address(0));
        }
        _transferOwnership(initialOwner);
    }

    /**
     * @dev Throws if called by any account other than the owner.
     */
    modifier onlyOwner() {
        _checkOwner();
        _;
    }

    /**
     * @dev Returns the address of the current owner.
     */
    function owner() public view virtual returns (address) {
        return _owner;
    }

    /**
     * @dev Throws if the sender is not the owner.
     */
    function _checkOwner() internal view virtual {
        if (owner() != _msgSender()) {
            revert OwnableUnauthorizedAccount(_msgSender());
        }
    }

    /**
     * @dev Leaves the contract without owner. It will not be possible to call
     * `onlyOwner` functions. Can only be called by the current owner.
     *
     * NOTE: Renouncing ownership will leave the contract without an owner,
     * thereby disabling any functionality that is only available to the owner.
     */
/*     
    // 
    function renounceOwnership() public virtual onlyOwner {
        _transferOwnership(address(0));
    }
*/
    /**
     * @dev Transfers ownership of the contract to a new account (`newOwner`).
     * Can only be called by the current owner.
     */
    function transferOwnership(address newOwner) public virtual onlyOwner {
        if (newOwner == address(0)) {
            revert OwnableInvalidOwner(address(0));
        }
        _transferOwnership(newOwner);
    }

    /**
     * @dev Transfers ownership of the contract to a new account (`newOwner`).
     * Internal function without access restriction.
     */
    function _transferOwnership(address newOwner) internal virtual {
        address oldOwner = _owner;
        _owner = newOwner;
        emit OwnershipTransferred(oldOwner, newOwner);
    }
}



//  -------- Stay below here --------------------------------------------------------
 


/**
 * @dev Implementation of the {IERC165} interface.
 *
 * Contracts that want to implement ERC165 should inherit from this contract and override {supportsInterface} to check
 * for the additional interface id that will be supported. For example:
 *
 * ```solidity
 * function supportsInterface(bytes4 interfaceId) public view virtual override returns (bool) {
 *     return interfaceId == type(MyInterface).interfaceId || super.supportsInterface(interfaceId);
 * }
 * ```
 */
abstract contract ERC165 is IERC165 {
    /**
     * @dev See {IERC165-supportsInterface}.
     */
    function supportsInterface(bytes4 interfaceId) public view virtual returns (bool) {
        return interfaceId == type(IERC165).interfaceId;
    }
}



//
// Base class that manages contract list
//
abstract contract GalleryV1Base is IGalleryV1 {

    //address of self that we return during the version check;
    address private m_currentGallery;

    // The following list is going to hold a collection of contracts that support
    // metadata hash functionality.
    address[] private m_ContractList;

    /**
     * @dev The constructor sets the initial state of the flip mapping, position zero is not used.
     */
    constructor()  {
        m_currentGallery = address(this);
    }

    // 
    function iAdd(address _contract) internal {
        m_ContractList.push(_contract);
    }
    /**
     * @dev Used to signal that there is a new Penny Oracle that should be used.
     */
    function iUpdateGallery(address _newGallery) internal  {
        m_currentGallery=_newGallery;
    }

    // gets the count of entries
    function countOf() external view returns (uint256 count) {
        return m_ContractList.length;
    }

    // returns the indexed contract
    function retrieve(uint256 _index) external view returns(address contractAddr) {
        require((_index < m_ContractList.length),"index out of range");
        return m_ContractList[_index];
    }

    /**
     * @dev Used to retrieve the address of the current Penny Oracle.
     */
    function version() external view returns (address) {
        return m_currentGallery;
    }

}



/*

Access security is handled by the main contract. Any routine that requires special access
rights is over written here and checked. 

*/
contract GalleryV1 is Ownable, GalleryV1Base, ERC165  {

    /**
     * @dev The constructor sets the initial state.
     */
    constructor() 
        Ownable(msg.sender)
        GalleryV1Base() 
        {
        }

    //
    // This routine is what adds the contract to the list. In order to be added to the list,
    // the _contract address must support IMetadataV1. Thus, this add routine calls to see if
    // it's supported. That address must support the IERC165 supportsInterface routine.
    //
    function Add(address _contract) external onlyOwner returns(bool bSuccess) {
        require(address(0) != _contract, "must provide valid contract");
        require(IERC165(_contract).supportsInterface(type(IMetadataV1).interfaceId),"needs to support IMetadataV1");
        require(IERC165(_contract).supportsInterface(type(IPayV1).interfaceId),"needs to support IPayV1");
        require(IERC165(_contract).supportsInterface(type(IERC721).interfaceId),"needs to support IERC721");

        iAdd(_contract);
        return true;
    }

    // If we put out a new gallery, we put that address here so it can be found by
    // anyone enumerating the gallery.
    function updateGallery(address _newGallery) external onlyOwner {
        require(address(0) != _newGallery, "must provide valid contract");
        iUpdateGallery(_newGallery);
    }


    /**
     * @dev See {IERC165-supportsInterface}.
     */
    function supportsInterface(bytes4 interfaceId) public view virtual override returns (bool) {
        return
            interfaceId == type(IGalleryV1).interfaceId ||
            super.supportsInterface(interfaceId);
    }
    // Publish our interface ID
    function InterfaceId() public pure returns(bytes4) {
        return type(IGalleryV1).interfaceId;
    }

}