// SPDX-License-Identifier: MIT
pragma solidity ^0.8.23;

import "INetworkV1.sol"; // interfaces.
import "IStandard.sol";  // standard interfaces
import "Library.sol"; // math, signedMath and Strings
import "abs_Ownable.sol";
import "abs_erc721.sol"; // main erc721
import "abs_MetaDataV1.sol";
import "abs_PayV1.sol";

/*
This registration class allows the NFT owner to register wallet addresses that can be seen
as being agents on the network. Because there are payouts associated with the NFT owner
that registers a new agent, we've got to track that relationship. When fetching the regsitered
address by id, it needs to return both the NFT owner and registered wallet addresses.

For clearity:

Founder: This is the owner of the agent NFT. This is the account that gets the finder reward.
         The founder address can change so it's linked to ownerOf(TokenId)
Partner: This is the registered agent that works on behalf of the Founder. Once registerd, these
         addresses are always static (unchangeable).

Both Founder and partners are considered Agents of the network. 

The partner agent ids are derrived from the token Id as in:

founderId = (tokenId * 1000) 
partnerId = founderId + (next available slot on TokenId)

This means that a agent id that has zeros for the last three digits will be treated as a
founder. ie, 12000 or 34000. Partner Ids will always have the founder Id with additional
numbers following like 12003 or 340012. 

When registering a new partner:

agentId = (TokenId * 1000) + (founder next slot)

When an agent NFT is transfered to a new owner, any access into the network using that
token id (founder) will be credited to the new owner. Yet, because all payouts are 
wallet based, any exiting cut that the previous owner had outstanding will still be
claimable by that previous wallet address owner. Thus, someone can setup a 'partnership system' that
is more valuable than a different token in the collection, thus the future performance of that NFT could
be more valuable then the future performance of another NFT in the collection.

Also note that all partner Ids are unique. Each one uses the token id along with a partner slot
id. This means that no two partner Ids can be the same. Thus no two agentIds are the same.

When splits are made, the Founder gets 10% and the Partner gets 90%. 

The Founder is allowed to register a fixed number of partners and no more.

*/
abstract contract RegisterV1 is ERC721, IRegisterV1, Ownable {

    uint256 private constant LIMIT_STEP = 42; // for each step along the way.
    uint256 private constant UPPER_LIMIT = (21*LIMIT_STEP); // 882 partners
    //uint256 private constant PARTNER_LIMIT = 42;
    // The partner limit for an NFT is initially set to 42. 
    //uint256 private m_founderLimit = 42;
    //uint256 private m_agentId;

    // given a founder id, this mapping returns the number of used slots. 
    mapping(uint256 founderId => uint256 count) private m_slots;
    // The initial limit to the number of partners will be 42. In order to extend that
    // regExtend() is called.
    mapping(uint256 founderId => uint256 limit) private m_limit;

    // Note that founders are not partners! when a partner id resolves to zero,
    // the code returns the founder wallet address. No partner Id can have the
    // least significant digits zero.

    // given an agent id, this returns the partner address
    mapping(uint256 Id => address agentAddr) private m_agentsById;
    // This mapping provides an Id for a wallet address. These partner Ids are fully linked
    // to the tokenId that registered them. 
    mapping(address partnerAddr => uint256 Id) private m_agentsByAddr;

    // We track the number of partners, every time someone is registered
    uint256 private m_totalAgents;

    // Every NFT minted has a token Id.

    constructor() {
        m_totalAgents = 0;
    }


    function underQuata(uint256 _founderIdAsTokenId) internal view returns(bool) {
        if( m_slots[_founderIdAsTokenId] < m_limit[_founderIdAsTokenId] ) {
            return true;
        }
        return false;
    }

    // Reduces agent id into the token id
    function founderTokenId(uint256 _agentId) internal pure returns(uint256 theFounderTokenId) {
        theFounderTokenId = (_agentId / 1000);
    }

    // This routine reduces the agentId down into it's parts and validates there are addresses
    function isAgentId(uint256 _agentId) public view returns(bool) {

        (address founderAddr, address partnerAddr) = iRegResolve(_agentId);
        if( founderAddr != address(0) && partnerAddr != address(0)) {
            return true;
        }
        return false;
    }
    // If this is a valid agent address, the Id will be returned, else zero.
    function isPartner(address _lookup) internal view returns(uint256 agentId) {
        return m_agentsByAddr[_lookup];
    }

    // if the lower section of the id is zero, we have a founder address
    function isFounderAgent(uint256 _agentId) internal pure returns(bool) {
        uint256 thePartnerId = _agentId - (founderTokenId(_agentId) * 1000);
        return ( 0 == thePartnerId );
    }

    // This routine is given the agent Id and it resolves the agent and the finder addresses.
    // The caller must validate that the agentId resolves to an existing token id.
    function iRegResolve(uint256 _agentId) internal view returns(address founderAddr, address partnerAddr) {
        
        // The founder address is always the owner of the token id
        founderAddr = ownerOf(founderTokenId(_agentId));

        if( isFounderAgent(_agentId) ) {
            partnerAddr = founderAddr;
        } else {
            partnerAddr = m_agentsById[_agentId];
        }
    }

    function iRegTotalAgents() internal view returns(uint256 totalAgents) {
        totalAgents = m_totalAgents;
    }

    // This routine sets the initial partner agent limit for the founder
    function iRegLimit(uint256 _founderIdIsTokenId) internal {

        m_limit[_founderIdIsTokenId] = LIMIT_STEP;
    }

    // All registration slots are based on token id. 
    function iRegAgent(uint256 _founderIdIsTokenId, address _partnerAddr) internal returns(uint256 thePartnerId) {

        // Each time a new agent is registered, we add to the slots used.
        m_slots[_founderIdIsTokenId] += 1;  

        // Generate the partner Id to return
        thePartnerId = (_founderIdIsTokenId * 1000) + m_slots[_founderIdIsTokenId];

        // record this new partner
        m_agentsById[thePartnerId] = _partnerAddr;
        m_agentsByAddr[_partnerAddr] = thePartnerId;
        m_totalAgents++;
    }

    // This routine reduces the agent Id to it's founder and returns the number of slots remaining.
    function regSlotsRemaining(uint256 _agentId) public view returns(uint256 count) {

        // reduce agent Id down into token id
        uint256 theFounderTokenId = founderTokenId(_agentId);
        require(theFounderTokenId != 0,"Invalid agent Id");
        require(_requireOwned(theFounderTokenId)!=address(0),"Token doesn't exist");
        // 
        count = (m_limit[theFounderTokenId] - m_slots[theFounderTokenId]);
    }

    // this routine extends the limit for the founder
    function iRegExtend(uint256 _theFounderTokenId) internal returns(uint256 count) {


        if( m_limit[_theFounderTokenId] <= UPPER_LIMIT ) {
            m_limit[_theFounderTokenId] += 42;
        }
        count = m_limit[_theFounderTokenId];
    }

}


/*

The Agent contract offers a commission for minting itself! This basically means that this contract
uses the PayV1 contract to handle payouts. Because we're not expecting to have to sell more than
one, the amount of NFTs that need to be sold are low(1). The 90 day timeout will still be used. 

*/
contract gv1Agent is MetaDataV1, PayV1, RegisterV1, ICountV1 {
    using Strings for uint256;

    uint256 private m_totalSupply; // Used to build the token Ids at mint time.
    uint256 private constant THREE_QUARTERS_DAY = (14400 * 3) / 4; // blocks represent time
    uint256 private m_lastMintBlock; // every time we mint, we record the block.

    //address private m_currentPennyOracle;
    address private m_participant; // must be participant in order to play
    address private m_website; // must have website nft to play

    // the commission payout constants if someone mints an agent
    uint256 private constant MIN_DISCOUNT_COUNT = 2;   // must mint 2 to get the commissions
    uint256 private constant AGENT_PERCENTAGE = 40;    // 40% of the mint price
    uint256 private constant WEBSITE_PERCENTAGE = 10;       // 10% of the mint price
    uint256 private constant INACTIVITY_DAYS = 180;    // must claim commissions within 180 days.

    uint256 private constant MINT_PENNIES = 15; // this value is stored in IPay
    uint256 private m_registrationPennies = 5;
    uint256 private m_sendLockPennies = 1;

    // fully transferable NFT with sendlock functionality
    bool private constant SOULBOUND = false;
    bool private constant SENDLOCK = true;

    /**
     * @dev The constructor sets the initial state of the flip mapping, position zero is not used.
     * @param _theTitled Address ultimate authority over contract. Account that receive funds.
     */
    constructor(address _theTitled,  address _thePennyOracle, address _theParticipant, address _theWebsite) 
        ERC721("AgentV1", "AGV1", SOULBOUND, SENDLOCK) 
        PayV1( _theTitled,  msg.sender, _thePennyOracle) {
            m_totalSupply=0;

            m_participant = _theParticipant;
            m_website = _theWebsite;

            // used for minting
            m_lastMintBlock = block.number;

            // Need to setup the commission payout structure in the PayV1 contract
            payInitalize(MINT_PENNIES, INACTIVITY_DAYS, MIN_DISCOUNT_COUNT, AGENT_PERCENTAGE, WEBSITE_PERCENTAGE);
    }

    // If this contract is setup and ready to work, this will return true.
    function IsReady() internal view returns(bool) {
        require(projectActive(),"setup Metadata, IsReady");
        require(payActive(),"Treasury not active, IsReady");
        if( 0 != m_totalSupply ) {
            require(m_website != address(0),"not setup yet, IsReady");
        }
        return true;
    }



    /**
     * @dev Used to set the penny prices for using this NFT. This should only be adjustable
     * by The Titled or the Owner of the contract.
     */
    function penniesSet(uint256 _mintPennies, uint256 _registrationPennies, uint256 _sendLockPennies) public onlyEither {
        iPayMintPenniesSet(_mintPennies);
        m_registrationPennies = _registrationPennies;
        m_sendLockPennies = _sendLockPennies;
    }

    /**
     * @dev Used to get the penny prices for using this NFT.
     */
    function penniesRead() public view returns (uint256 mint, uint256 registration, uint256 sendLock) {
        mint = iPayMintPenniesRead();
        registration = m_registrationPennies;
        sendLock = m_sendLockPennies;
    }

    /**
     * @dev Used to get the wei prices for using this NFT.
     */
    function pricesRead() public view returns(uint256 mintWei, uint256 registrationWei, uint256 sendlockWei) {
        mintWei = currentWei(iPayMintPenniesRead());
        registrationWei = currentWei(m_registrationPennies);
        sendlockWei = currentWei(m_sendLockPennies);
    }

    //  ------------------  IPennyOracleV1 interface ------------------ 

    function pennyOracleSet(address _PennyOracle) public onlyOwner {
        require(_PennyOracle != address(0), "No Penny to Reference");
        require(IERC165(_PennyOracle).supportsInterface(type(IPennyOracleV1).interfaceId),"doesn't support IPennyOracleV1");
        iPennyOracleSet(_PennyOracle);
    }



    // the registration process needs to know how much it will cost
    //function sendLockPrice() public view returns(uint256 priceWei) {
    //    priceWei = currentWei(m_sendLockPennies);
    //}
    // can only setup a lock on an existing toke that the caller owns. 
    function sendLockRegister(address to, uint256 tokenId) public payable {
        _requireOwned(tokenId);
        require(_ownerOf(tokenId) == msg.sender, "Must own token to sendlock it");
        require(to != msg.sender, "Don't send to self");
        require(address(0) != to, "must provide valid destination");
        require(currentWei(m_sendLockPennies) == msg.value, "Wrong amount sent");

        // record the payment for The Titled
        titledPayment();

        _sendLockRegister(to,tokenId);
    }


    //  ------------------  IMetadataV1 interface ------------------ 

    /**
     * @dev This routine sets the initial state for the project metadata. The _URI is used
     * as the base URI for all token URIs. The _Project name represents the file name(without
     * extension) on the server pointed to by _URI. The _Project JSON file can be hashed to
     * produce the SHA1 hash value represented by _hash. Only the owner of the contract can
     * set the project location. If the project moves, the latest addition will be considered
     * the working location.
     */
    function projectUpdate(string calldata _URI, string calldata _Project, string calldata _hash) public onlyOwner {
        require(bytes(_URI).length > 0,"Project base URI needs valid path");
        require(bytes(_Project).length > 0,"Project name length invalid");
        require(bytes(_hash).length == 40,"requires SHA1 hash string");
        return iProjectUpdate(_URI,_Project,_hash);
    }

    // ------------------ ERC721 functionality ------------------ 

    /**
     * @dev returns the appropriate metadata URI for the given token id. Note that the token id
     * must exist in order for it to be built and returned. 
     */
    function tokenURI(uint256 tokenId) public view override returns (string memory) {
        _requireOwned(tokenId);
        string memory resultURI = string.concat(_baseURI(), "agentv1.json");
        return resultURI;
    }

    /**
     * @dev Base URI for computing {tokenURI}. 
     */
    function _baseURI() internal view override(ERC721) returns (string memory) {
        string memory resultURI = string.concat(projectBaseURI(),"metadata/");
        return resultURI;
    }

    // In order to mint, we have to make that functionality public. We will do so with a
    // function called payMint() that is payable. If the checks are passed, we will call the
    // safeMint override in order to track the total minting. 
    function safeMint(address to, address _founder, address _partner, address _website) internal returns(uint256 tokenId) {
        m_totalSupply++; // used to id the tokens. 

        // set limit for founder
        iRegLimit(m_totalSupply);

        // only place where funds are added to the treasury
        mintSplit(_founder, _partner, _website);

        m_lastMintBlock = block.number; // used to calculate rest between mintings.
        _safeMint(to, m_totalSupply);
        return m_totalSupply;
    }

    //  ------------------ Used for ICountV1 Interface  ------------------ 

    /**
     * @dev Anyone is allowed to see the membership totals
     */
    function totalSupply() external view returns(uint256 theTotalSupply) {
        return m_totalSupply;
    }


    //  ------------------ Used for IPayV1 Interface  ------------------ 

    // There are only two ways to withdraw and they both come through here.
    function iWithdraw(address _anAccount) internal returns(bool bSuccess) {

        uint256 uiWei = claimAccount(_anAccount);
        payable(_anAccount).transfer(uiWei);
        return true;
    }


    // export the number of agents
    function regTotalAgents() public view returns(uint256 totalAgents) {
        return iRegTotalAgents();
    }

    // This routine returns true if the conditions for minting are met.
    // The first 21 NFTs are openly mintable
    // after that, up to 42 can only be 1 per day.
    function IsMintReady() public view returns(bool) {
        if( m_totalSupply < 21 ) {
            return true;
        }

        // Everything over 21 requires at least 3/4ths day rest between mints
        if( (m_lastMintBlock + THREE_QUARTERS_DAY) > block.number ) {
            return false; // the delay hasn't been long enough since last mint.
        }

        // just a delay for the second 21
        if( m_totalSupply < 42 ) {
            return true;
        }

        // Now that we have at least 42 founder agents, we're going to require that these agents
        // register partners before more minting is allowed.

        // A complication is that if an investor allocates and sits on a bunch of agent
        // nfts, the network will be stunted and at the mercy of the NFT holder. Thus
        // to incentivize the registration of partners, any founder that fills their
        // NFT with at least 38 partners will be allowed extend their partner limit, but
        // only if the following check is true.    
        
        // To move forward, we will only mint if the number of agents is
        // greater than 8 per NFT. At minimum, 336 agents at 42 NFTs. 
        uint256 totalAgents = iRegTotalAgents();
        if( totalAgents < (m_totalSupply * 8) ) {
            return false;
        }

        // One last check. The number of NFTs will be proportional to the number of participants 
        // in the network. require 100 participants per agent

        uint256 totalParticipants = ICountV1(m_participant).totalSupply();
        if( (totalParticipants/100) >= totalAgents ) {
            return true;
        }
        // else, need to grow the number of participants on hte network.
        return false;
    }

    // Under very special conditions, a founder can extend the number of partners allowed
    // for the token.
    function regExtend(uint256 _agentId) public returns(uint256 count) {


        // reduce agent Id down into token id
        uint256 theFounderTokenId = founderTokenId(_agentId);
        require(theFounderTokenId != 0,"Invalid agent Id");
        require(_requireOwned(theFounderTokenId)!=address(0),"Token doesn't exist");

        (address founderAddr, ) = iRegResolve(_agentId);
        require(msg.sender == founderAddr,"must be founder to extend");

        // The check we perform is:
        // - if there are at least 42 tokens &
        // - the total number of agents is low &
        // - the NFT is full
        // then, allow an extension. 

        require(m_totalSupply >= 42,"Still within bounds");
        require(iRegTotalAgents()<(m_totalSupply * 8),"to many agents");
        require(regSlotsRemaining(_agentId) < 5,"fill more slots");

        count = iRegExtend(theFounderTokenId);
    }


    /**
     * @dev Anyone is allowed to mint provided the metadata information has already been
     * setup, the right amount of coin is provided. There is no limit to the 
     * number of NFTs that can be minted. The caller must provide two Ids, the first one
     * is the agent Id, the second is the website Id. Both Ids must be in good standing with
     * the community to pass the minting check.
     */
    function payMint(uint256 _agentId, uint256 _websiteId) public payable returns(uint256 tokenId) {

        require(IsReady(),"Not setup yet, Mint");
        require(payValidateAmount(), "Wrong amount sent, Mint");

        // must hold a participant NFT in order to play.
        require( IERC721(m_participant).balanceOf(msg.sender) > 0, "must hold participant to play");

        // Because we are incentivizing the minting of our own NFT, we need supply all three
        // payout wallets
        address cutFounderAddr;
        address cutPartnerAddr;
        address cutWebsiteAddr;

        // At NFT launch we will not have any agent NFts to check against. Until that first one
        // exists, we have to special case this check.
        if( 0 == m_totalSupply ) {
            // allow the smart contract owner to mint the first agent. 
            require(owner() == msg.sender, "unauthorized minting");
            require(0 == _agentId, "agent id incorrect");
            require(0 == _websiteId, "website id incorrect");
            //TODO: Mint commissions go to owner()'s address
            cutFounderAddr = owner();
            cutPartnerAddr = owner();
            cutWebsiteAddr = owner();

            // now that we've cleared the checks, we have to register ourselve so the founder and partner
            // checks will pass. We grab the future token Id and use that to register this first founder
            // as a partner. Thus, this first NFT should have a founder agentId of 1000 or, of the 
            // NFT is ever transfered, the project founder will be able to use 1001 as a partner.
            iRegAgent((m_totalSupply+1),msg.sender);

        } else {
            // Are we ready to mint?
            require(m_website != address(0),"Not Setup");
            // Validate that the minting code is crediting a website NFT holder
            cutWebsiteAddr = IERC721(m_website).ownerOf(_websiteId);
            require(cutWebsiteAddr != address(0),"invalid website Id");

            (cutFounderAddr,cutPartnerAddr) = iRegResolve(_agentId);
            require(cutFounderAddr != address(0),"Invalid agent Id");
            require(cutPartnerAddr != address(0),"Invalid agent Id");

            //TODO: minting is restricted. Perform that check here.
            require(IsMintReady() == true, "Not qualified for mint");
        }

        // At this point, info looks valid, allow mint
        return safeMint(msg.sender, cutFounderAddr, cutPartnerAddr, cutWebsiteAddr);
    }


    // This routine will be called by either the contract owner or The Titled in order to 
    // capture the outstanding commission from a deliquent agent. Note that the funds are
    // assigned to The Titled. 
    function paySweep(address _anAccount) external onlyEither returns(bool) {
        
        require(sweepConditions(_anAccount), "account not ready");
        return updateAbandonedAccount(_anAccount);
    }

    /**
     * @dev If you have an account, you can get to your funds if all conditions are met
     */
    function payWithdraw() external returns(bool) {

        // has to pass the checks in order to withdraw.
        address recieverAddr = msg.sender;
        require(claimConditions(recieverAddr),"conditions not met");
        return iWithdraw(recieverAddr);
    }
    function payWithdrawTitled() external onlyEither returns(bool) {

        // has to pass the checks in order to withdraw.
        address recieverAddr = titled();
        require(claimConditions(recieverAddr),"conditions not met");
        return iWithdraw(recieverAddr);
    }

    //  ------------------  IRegisterV1 interface ------------------ 


    // This routine is used to register a new wallet as an agent in the network. It requires
    // payment of REGISTRATION_PENNIES amount of tfuel. An agent can only be added once. 
    function regAgent(uint256 _agentId, address _partnerAddr) public payable returns(uint256 partnerId) {
        // they have to send the correct amount of tfuel
        require(currentWei(m_registrationPennies) == msg.value, "incorrect registration value");

        // the account calling (msg.sender) needs to be a NFT holder of the founderId
        uint256 founderIdIsTokenId = founderTokenId(_agentId);
        require(msg.sender == _ownerOf(founderIdIsTokenId),"doesn't own founder nft");

        // the founder must not have exceeded his registration quota
        require(underQuata(founderIdIsTokenId)==true,"exceeded quota");

        //TODO: can't add an agent that doesn't hold a participant NFT
        require( IERC721(m_participant).balanceOf(_partnerAddr) > 0, "must hold participant to play");

        // Lastely, once an address is registered, it can't be registered a second time.
        require(isPartner(_partnerAddr) == 0,"Partner already registered");

        // record the payment for The Titled
        titledPayment();
        // Now pass our validated info to the registration handler
        return iRegAgent(founderIdIsTokenId,  _partnerAddr);
    }

    // used to confirm that a wallet is a registered partner. Owners of an NFT are 
    // founders.
    function regLookupPartner(address _lookup) public view returns(uint256 agentId) {

        return isPartner(_lookup);
    }

    // This routine is called by anyone implimenting the IPay interface. IPay requires that
    // the agent Id is broken down into the payable wallets.
    function regResolve(uint256 _agentId) public view returns(address founderAddr, address partnerAddr) {

        (founderAddr,partnerAddr) = iRegResolve(_agentId);
        // if either one of the addresses do not resolve, we will return both as bad.
        if( founderAddr == address(0) || partnerAddr==address(0)) {
            founderAddr = address(0);
            partnerAddr = address(0);
        }
    }

    //  ------------------  IERC165 interface ------------------ 

    //
    // When this NFT project is added to The Gallery V1, that contract will query
    // to make sure this NFT supports IMetadataV1 and IPayV1. It will not accept
    // NFT projects that don't offer this functionality.
    //
    /**
     * @dev See {IERC165-supportsInterface}.
     */
    function supportsInterface(bytes4 interfaceId) public view virtual override returns (bool) {
        return
            interfaceId == type(IMetadataV1).interfaceId ||
            interfaceId == type(IPayV1).interfaceId ||
            interfaceId == type(IRegisterV1).interfaceId ||
            interfaceId == type(ICountV1).interfaceId ||
            interfaceId == type(IPennyOracleV1).interfaceId ||
            super.supportsInterface(interfaceId);
    }

}