// SPDX-License-Identifier: MIT
pragma solidity ^0.8.23; //4; //23;

import "inetworkv1.sol"; // holds all interfaces
import "istandard.sol"; // all standard Interfaces
import "library.sol"; // Math, signedMath and Strings
import "abs_ownable.sol";

import "abs_metadatav1.sol"; 
import "abs_erc721.sol"; // main erc721
import "abs_payv3.sol"; // slight mod from v2 & v1

/*
minting: in order to mint, the caller is going to have to specify both a series and spot number that
will be associated with each token. When the token URI is built, the code will reference the metadata
like: ./projectName_series_spot.json. 

When fetching the price, we will need both the series and spot numbers.

The project owner should be able to add a new series and then add new spot items to it. When creating
a new series & spot entry, the creator presents the price (in pennies) for that item. Likewise, the 
creator also determines the minting limit.
*/
/*

    // From ICardsv1
    // Only callable by owner
    function cardNewSpot(uint256 _seriesNum, uint256 _spotNum, uint256 _pennies, uint256 _limit) external;
    function cardPenniesWrite(uint256 _seriesNum, uint256 _spotNum,uint256 _pennies) external;
    function cardMintToggle(uint256 _seriesNum, uint256 _spotNum) external;
    // allow the owner to mint without commissions
    function cardOwnerMint(address _recipient, uint256 _seriesNum, uint256 _spotNum) external returns(uint256 tokenId);

    // can be accessed by anyone
    function cardSeriesCount() external view returns(uint256 count);
    function cardSpotCount(uint256 _seriesNum) external view returns(uint256 count);
    function cardInfo(uint256 _seriesNum, uint256 _spotNum) external view returns(uint256 pennies, uint256 limit, uint256 total, bool bMintable);
    function cardPenniesRead(uint256 _seriesNum, uint256 _spotNum) external view returns(uint256 pennies);
    function cardMintPrice(uint256 _seriesNum, uint256 _spotNum) external view returns(uint256 mintTfuelWei);
    function cardMint(uint256 _AgentId, uint256 _websiteId, uint256 _seriesNum, uint256 _spotNum) external payable returns(uint256 tokenId);

    function cardSeriesSpot(uint256 _tokenId) external view returns(uint256 seriesNum, uint256 spotNum);

    // from PayV1
    what is needed:
    function payInitalize(uint256 _mintPennies, uint256 _timeoutDays, uint256 _minCount, uint256 _cutPercentAgent, uint256 _cutPercentWebsite) external;
    function payMintInfo() external view returns (uint256 payAgent, uint256 payWebsite, uint256 delayBlocks, uint256 minMints);
    function payWithdraw() external returns(bool);
    function payWithdrawTitled() external returns(bool);
    function paySweep(address _anAccount) external returns(bool);
    function payTreasuryBalance() external view returns(uint256 BalanceWei);
    function payAgentBalance(address _account) external view returns(uint256 payBalanceWei, uint256 payDeadline, uint256 payCount);
    function payActive() external view returns (bool);

    // payBurn functionality
    function payBurnPrice() external view returns(uint256 burnWei);
    function payBurn(uint256 _tokenId) external payable;
    // sendLock functionality
    function paySendLockPrice() external view returns(uint256 sendLockWei);
    function sendLockRegister(address to, uint256 tokenId) public payable;

    what can be skipped:
    function payMint(uint256 _AgentId, uint256 _websiteId) external payable returns(uint256 tokenId);
    function payMintPrice() external view returns(uint256 mintTfuelWei);

    // from IERC721ExV1
    function IsSendLock() external returns (bool bSendLock);
    function paySendLockPrice() external view returns(uint256 sendLockWei);
    function sendLockRegister(address to, uint256 tokenId) external payable;
    function IsSoulBound() external returns (bool bSoulBound);
    function payBurnPrice() external view returns(uint256 burnTfuelWei);
    function payBurn(uint256 _tokenId) external payable ;
*/

//
// Main class
//
contract CARDV1 is IERC721ExV1, PayV1, MetaDataV1,  ICountV1, ICardsV1, IMintV1   {
    using Strings for uint256;

    //uint8 private constant MINT_INDEX = 0;
    uint8 private constant BURN_INDEX = 1;
    uint8 private constant SENDLOCK_INDEX = 2;

    uint256 private m_totalSupply;
    uint256 private m_totalBurnt;
    // If m_limitedSupply is non-zero, this contract will toggle minting off as soon as the
    // limit is reached. Set in constructor.
    //uint256 private m_limitedSupply; 

    address private m_participant; // must be participant in order to play
    // During mint, we call these two different contracts to make sure the two addresses that are provided
    // both hold ligitimate NFTs of the project. 
    address private m_agent = address(0);
    address private m_website = address(0);

    string private constant theSymbol = "CARDV1";

    // the owner of this contract can assign token based metadata.
    mapping(uint256 tokenId => bool bOn) private m_bCustomURL;

    // used to enable and disable minting for the entire project. If disabled,
    // nothing can be minted.
    bool private m_bMintToggle = true; // allows titled & owner to enable/disable minting for everything.

    // for cards functionality ----

    // the tok structure holds the series and spot for the individual token. 
    struct tok {
        uint256 s_series;
        uint256 s_spot;
    }
    mapping(uint256 tokenId => tok) private m_toks;

    struct spot {
        uint256 s_pennies; // cost for minting
        uint256 s_limit;   // how many can be minted
        uint256 s_total;   // count of number minted.
        bool s_bMintable;  // can this be minted?
    }
    struct series {
        uint256 s_spotCount; // number of spots for series
        mapping(uint256 spotNum => spot) m_spots; // mapping of spot structures
        mapping(uint256 spotNum => bool bInUse) m_Numbers; // mapping of spot numbers.
    }
    uint256 private m_seriesCount = 0;
    mapping(uint256 seriesNum => bool bInUse) private m_SeriesNumbers;  // mapping of series numbers.
    mapping(uint256 seriesNum => series ) private m_cards; // Index to structure



    // This performs the actual setting of the values. We're expecting that
    // we'll get a penny value and limit to how many can be minted. If limit is
    // zero, it's an unlimted mint.
    function iCardNewSpot(uint256 _seriesNum, uint256 _spotNum, 
            uint256 _pennies, uint256 _limit) internal {

        series storage seriesItem = m_cards[_seriesNum];
        // The spot hasn't been used. We create a new one

        seriesItem.m_spots[_spotNum].s_pennies = _pennies;
        seriesItem.m_spots[_spotNum].s_limit = _limit;
        seriesItem.m_spots[_spotNum].s_total = 0;
        seriesItem.m_spots[_spotNum].s_bMintable = true;

        // We have a new spot on this series
        seriesItem.s_spotCount++;
        seriesItem.m_Numbers[_spotNum] = true;
    }
    function cardMintToggle(uint256 _seriesNum, uint256 _spotNum) external onlyEither {

        require(true == m_SeriesNumbers[_seriesNum], "Series doesn't exist");
        series storage seriesItem = m_cards[_seriesNum];
        require(true == seriesItem.m_Numbers[_spotNum],"Spot doesn't exist");
        // we've validated the series and spot numbers are recorded.
        if( seriesItem.m_spots[_spotNum].s_bMintable ) { 
            seriesItem.m_spots[_spotNum].s_bMintable = false; 
        } else { 
            seriesItem.m_spots[_spotNum].s_bMintable = true; 
        }
    }

    function cardSeriesCount() public view returns(uint256 count) {
        return m_seriesCount;
    }
    function cardSpotCount(uint256 _seriesNum) public view returns(uint256 count) {
        count=0;
        if( true == m_SeriesNumbers[_seriesNum] ) {
            series storage seriesItem = m_cards[_seriesNum];
            count = seriesItem.s_spotCount;
        }
        return count;
    }

    // The caller is expected to know the series and spot numbers for what they are
    // creating. The default state is mintable.
    function cardNewSpot(uint256 _seriesNum, uint256 _spotNum, 
            uint256 _pennies, uint256 _limit) public onlyEither {
        require(_pennies>0,"Must have value");

        uint256 SeriesNum = _seriesNum;
        uint256 SpotNum = _spotNum;

        // If caller provides a series number that we already have, we try to use it.
        if( true == m_SeriesNumbers[SeriesNum] ) {
            // the series exists, what about teh spot? If spot exists, we bail.
            series storage seriesItem = m_cards[SeriesNum];
            require(false == seriesItem.m_Numbers[SpotNum],"Numbers in use");

            // spot doesn't exist, we can create this.
            iCardNewSpot(SeriesNum, SpotNum, _pennies, _limit);

        } else {
            // we can create this spot, when we do, we register the new series..
            iCardNewSpot(SeriesNum, SpotNum, _pennies, _limit);

            m_SeriesNumbers[SeriesNum] = true;
            m_seriesCount++;
        }
    }

    // When querying for card info, if all the numbers are zero, it's an empty location.
    // So, we don't have to validate if the item exists or not. 
    function cardInfo(uint256 _seriesNum, uint256 _spotNum) public view returns(uint256 pennies, uint256 limit, uint256 total, bool bMintable) {
        series storage seriesItem = m_cards[_seriesNum];
        pennies = seriesItem.m_spots[_spotNum].s_pennies;
        limit = seriesItem.m_spots[_spotNum].s_limit;
        total = seriesItem.m_spots[_spotNum].s_total;
        bMintable = seriesItem.m_spots[_spotNum].s_bMintable;
        return (pennies, limit, total, bMintable);
    }

    function iCardPenniesRead(uint256 _seriesNum, uint256 _spotNum) internal view returns(uint256 pennies) {

        require(true == m_SeriesNumbers[_seriesNum], "Series doesn't exist");
        series storage seriesItem = m_cards[_seriesNum];
        require(true == seriesItem.m_Numbers[_spotNum],"Spot doesn't exist");
        // we've validated the series and spot numbers are recorded.
        pennies = seriesItem.m_spots[_spotNum].s_pennies;
        return (pennies);
    }

    // reads the number of pennies it takes to mint this series spot 
    function cardPenniesRead(uint256 _seriesNum, uint256 _spotNum) external view returns(uint256 pennies) {
        return iCardPenniesRead(_seriesNum, _spotNum);
    }
    // only if the series & spot exist will we get a Wei price.
    function cardMintPrice(uint256 _seriesNum, uint256 _spotNum) external view returns(uint256 mintTfuelWei) {
        uint256 pennies = iCardPenniesRead(_seriesNum, _spotNum);
        mintTfuelWei = currentWei(pennies);
        return (mintTfuelWei);
    }

    // standard conditions for minting.
    function iCardSafeMint(address to, address _founder, address _partner, address _website, uint256 _series, uint256 _spot) internal returns(uint256 tokenId) {
        m_totalSupply++; // used to id the tokens. 

        mintSplit(_founder, _partner, _website, m_cards[_series].m_spots[_spot].s_pennies);

        _safeMint(to, m_totalSupply);

        // Every token has its series and spot numbers.
        m_toks[m_totalSupply] = tok(_series,_spot);
        // Every series & spot has a running total
        m_cards[_series].m_spots[_spot].s_total++;

        m_bCustomURL[m_totalSupply]=false; // no custom metadata at start
        return m_totalSupply;
    }
    // owner can mint with no commissions.
    function iCardSafeOwnerMint(address to, uint256 _series, uint256 _spot) internal returns(uint256 tokenId) {
        m_totalSupply++; // used to id the tokens. 

        _safeMint(to, m_totalSupply);

        // Every token has its series and spot numbers.
        m_toks[m_totalSupply] = tok(_series,_spot);
        // Every series & spot has a running total
        m_cards[_series].m_spots[_spot].s_total++;

        m_bCustomURL[m_totalSupply]=false; // no custom metadata at start
        return m_totalSupply;
    }

    // owner can adjust the price of this offering.
    function cardPenniesWrite(uint256 _seriesNum, uint256 _spotNum,uint256 _pennies) external onlyEither {
        require(true == m_SeriesNumbers[_seriesNum], "Series doesn't exist");
        series storage seriesItem = m_cards[_seriesNum];
        require(true == seriesItem.m_Numbers[_spotNum],"Spot doesn't exist");
        // we've validated the series and spot numbers are recorded. Now update
        // the number of pennies it takes to mint this item.
        seriesItem.m_spots[_spotNum].s_pennies = _pennies;
    }

    function cardMint(uint256 _AgentId, uint256 _websiteId, uint256 _seriesNum, uint256 _spotNum) external payable returns(uint256 tokenId) {

        require(imintable()==true,"not mintable"); // ownerLockToggle    
        require(IsReady(),"Not setup yet"); // can't mint until minting commissions are set
        require(iPayActive(),"set up commissions");

        // is the series spot mintable?
        require(true == m_SeriesNumbers[_seriesNum], "Series doesn't exist");
        series storage seriesItem = m_cards[_seriesNum];
        require(true == seriesItem.m_Numbers[_spotNum],"Spot doesn't exist");
        // we've validated the series and spot numbers are recorded.
        require(true == seriesItem.m_spots[_spotNum].s_bMintable,"not mintable");

        if( 0 != seriesItem.m_spots[_spotNum].s_limit ) {
            // there is an upper limit, are we still below it?
            require(seriesItem.m_spots[_spotNum].s_total <= seriesItem.m_spots[_spotNum].s_limit,"At limit");
        }
        // Looks like this item is mintable, did the user send enough?
        uint256 mintTfuelWei = currentWei(iCardPenniesRead(_seriesNum, _spotNum));
        require(mintTfuelWei==msg.value, "Wrong amount sent");

        // Did they provide the infrastructure?
        require( IERC721(m_participant).balanceOf(msg.sender) > 0, "must hold participant");

        // Validate that the minting code is crediting a website NFT holder
        address cutWebsiteAddr = IERC721(m_website).ownerOf(_websiteId);
        require(cutWebsiteAddr != address(0),"invalid website Id");

        // now resolve the agent id
        (address cutFounderAddr, address cutPartnerAddr) = IRegisterV1(m_agent).regResolve(_AgentId);
        require(cutFounderAddr != address(0),"Invalid agent Id");
        require(cutPartnerAddr != address(0),"Invalid agent Id");

        // Any other restrictions on minting? 
        return iCardSafeMint(msg.sender, cutFounderAddr, cutPartnerAddr, cutWebsiteAddr,_seriesNum,_spotNum);
    }
    function cardOwnerMint(address _recipient, uint256 _seriesNum, uint256 _spotNum) external onlyEither returns(uint256 tokenId) {

        // is the series spot mintable?
        require(true == m_SeriesNumbers[_seriesNum], "Series doesn't exist");
        series storage seriesItem = m_cards[_seriesNum];
        require(true == seriesItem.m_Numbers[_spotNum],"Spot doesn't exist");
        // we've validated the series and spot numbers are recorded.

        // let the owner mint regardless of the limits 
        return iCardSafeOwnerMint(_recipient, _seriesNum, _spotNum);
    }

    function cardSeriesSpot(uint256 _tokenId) external view returns(uint256 seriesNum, uint256 spotNum) {
        _requireOwned(_tokenId);
        return ( m_toks[_tokenId].s_series,m_toks[_tokenId].s_spot);
    }

    /**
     * @dev The constructor sets the initial state of the flip mapping, position zero is not used.
     * @param _theTitled This is the project owner's wallet. Considered the owner of the project
     * @param _thePennyOracle The Penny Oracle smart contract address
     */
    constructor(address _theTitled, 
                address _thePennyOracle, 
                address _theParticipant, 
                address _theAgent, 
                address _theWebsite,
                string memory _strProjectName,
                bool _bSoulbound,
                bool _bSendLock
                ) 
        ERC721(_strProjectName, theSymbol, _bSoulbound, _bSendLock) 
        PayV1( _theTitled,  msg.sender, _thePennyOracle)
        {
            m_participant = _theParticipant;
            m_agent = _theAgent;
            m_website = _theWebsite;
            //m_limitedSupply = _limit;
        }

    /**
     * @dev Used to set price and commission info. This should only be set by the Titled or
     * owner of the contract. This should be the first item set after launching the contract.
    */
    function payInitalize(uint256 _mintPennies, uint256 _timeoutDays, uint256 _minCount, uint256 _cutPercentAgent, uint256 _cutPercentWebsite) external onlyEither {
        require(iPayActive()==false,"Already setup");
        // validate that both counts add to less then 100
        require(((_cutPercentAgent+_cutPercentWebsite)<100),"bad math");
        // NOTE: _mintPennies is ignored in this contract.
        iPayInitalize(_mintPennies, _timeoutDays, _minCount, _cutPercentAgent, _cutPercentWebsite);

        // Set the number of pennies for this other activity.
        iPayPenniesSet(BURN_INDEX, 0);
        iPayPenniesSet(SENDLOCK_INDEX, 10);
    }

    /**
     * @dev Used to set the penny price for minting the NFT. This should only be adjustable
     * by The Titled or the Owner of the contract.
     */
    function penniesSet(uint256 /* unused */) public onlyEither {
    }
    /**
     * @dev Used to get the penny price for using this NFT.
     */
    function penniesRead() public pure returns(uint256 /* usused */) {
        return 0;
    }

    // If this contract is setup and ready for minting, this will return true.
    function IsReady() internal view returns(bool) {
        require(m_agent != address(0), "not setup");
        require(projectActive(),"Metadata not setup");
        require(true == m_bMintToggle,"mint disabled");
        return true;
    }

    // ------------------ IMintV1 functionality ------------------ 
    //
    // For the cards project, it can be abled/disabled via the ownerLockToggle().

    function imintable() internal view returns(bool bMintable) {
        return m_bMintToggle; // return what the project owner wants.
    }
    function mintable() external view returns(bool bMintable)
    {
        return imintable();
    }
    function ownerLockToggle() external onlyEither {
        if( m_bMintToggle ) { m_bMintToggle = false; } else { m_bMintToggle = true; }
    }

    // ------------------ ERC721 functionality ------------------ 

    // Used when assigning a custom metadata file to a specific token id.
    function customMetadataToggle(uint256 _tokenId) external onlyEither {
        if( true == m_bCustomURL[_tokenId] ) {
            m_bCustomURL[_tokenId] = false;
        } else {
            m_bCustomURL[_tokenId] = true;
        }
    }
    
    /**
     * @dev returns the appropriate metadata URI for the given token id. Note that the token id
     * must exist in order for it to be built and returned. 
     */
    //
    //  want baseURI projectName_seriesNum_spotNum.json as the default. If the user
    // gets custom metadata, we add the token id.
    //
    function tokenURI(uint256 _tokenId) public view override returns (string memory) {
        _requireOwned(_tokenId);

        uint256 seriesNum = m_toks[_tokenId].s_series;
        uint256 spotNum = m_toks[_tokenId].s_spot;

        string memory projectName;
        (, projectName, ) = projectDataCurrent();

        string memory resultURI;
        resultURI = string.concat(_baseURI(), projectName, "_", Strings.toString(seriesNum) , "_", Strings.toString(spotNum));
        if( true == m_bCustomURL[_tokenId] ) {
            resultURI = string.concat(resultURI, "_", Strings.toString(_tokenId), ".json");
        } else {
            resultURI = string.concat(resultURI, ".json");
        }
        return resultURI;
    }

    /**
     * @dev Base URI for computing {tokenURI}. 
     */
    function _baseURI() internal view override(ERC721) returns (string memory) {
        string memory resultURI = string.concat(projectBaseURI(),"metadata/");
        return resultURI;
    }

    //  ------------------ Used for IPayV1 Interface  ------------------ 

    function payMintPrice() external pure returns(uint256 /* unused */) {
        return 0;
    }

    /**
     * @dev Anyone is allowed to mint provided the metadata information has already been
     * setup, the right amount of coin is provided. There is no limit to the 
     * number of NFTs that can be minted. The caller must provide two wallets. Both 
     * wallets must hold agent NFTs
     */
    function payMint(uint256 /* unused*/, uint256 /* unused*/) public payable returns(uint256 /* unused */) {
        return 0;
    }

    // This routine will be called by either the contract owner or The Titled in order to 
    // capture the outstanding commission from a deliquent agent. Note that the funds are
    // assigned to The Titled. 
    function paySweep(address _anAccount) external onlyEither returns(bool) {
        
        require(sweepConditions(_anAccount), "account not ready");
        return updateAbandonedAccount(_anAccount);
    }

    // There are only two ways to withdraw and they both come through here.
    function iWithdraw(address _anAccount) internal returns(bool bSuccess) {

        uint256 uiWei = claimAccount(_anAccount);
        payable(_anAccount).transfer(uiWei);
        return true;
    }
    /**
     * @dev If you have an account, you can get to your funds if all conditions are met
     */
    function payWithdraw() external returns(bool) {

        // has to pass the checks in order to withdraw.
        address recieverAddr = msg.sender;
        require(claimConditions(recieverAddr),"conditions not met");
        return iWithdraw(recieverAddr);
    }
    function payWithdrawTitled() external onlyEither returns(bool) {

        // has to pass the checks in order to withdraw.
        address recieverAddr = titled();
        require(claimConditions(recieverAddr),"conditions not met");
        return iWithdraw(recieverAddr);
    }

    //  ------------------  IMetadataV1 interface ------------------ 

    /**
     * @dev This routine sets the initial state for the project metadata. The _URI is used
     * as the base URI for all token URIs. The _Project name represents the file name(without
     * extension) on the server pointed to by _URI. The _Project JSON file can be hashed to
     * produce the SHA1 hash value represented by _hash. Only the owner of the contract can
     * set the project location. If the project moves, the latest addition will be considered
     * the working location.
     */
    function projectUpdate(string calldata _URI, string calldata _Project, string calldata _hash) public onlyOwner {
        require(bytes(_URI).length > 0,"Project base URI needs valid path");
        require(bytes(_Project).length > 0,"Project name length invalid");
        require(bytes(_hash).length == 40,"requires SHA1 hash string");
        return iProjectUpdate(_URI,_Project,_hash);
    }

    //  ------------------  IPennyOracleV1 interface ------------------ 
    //
    // The Penny Oracle functionality is handled by the IPay class.
    //
    // This NFT uses the Penny Oracle. It is set in the constructor, but if it changes
    // we can update that source here.
    //
    function pennyOracleSet(address _PennyOracle) public onlyOwner {
        require(_PennyOracle != address(0), "No Penny to Reference");
        require(IERC165(_PennyOracle).supportsInterface(type(IPennyOracleV1).interfaceId),"doesn't support IPennyOracleV1");
        iPennyOracleSet(_PennyOracle);
    }

    //  ------------------  ICountV1 interface ------------------ 

    /**
     * @dev Anyone is allowed to see the totals
     */
    function totalSupply() external view returns(uint256 theTotalSupply) {
        return m_totalSupply-m_totalBurnt;
    }


    //  ------------------  IERC165 interface ------------------ 
    //
    // When this NFT project is added to The Gallery V1, that contract will query
    // to make sure this NFT supports IMetadataV1 and IPayV1. It will not accept
    // NFT projects that don't offer this functionality.
    //
    /**
     * @dev See {IERC165-supportsInterface}.
     */
    function supportsInterface(bytes4 interfaceId) public view virtual override returns (bool) {
        return
            interfaceId == type(IPennyOracleV1).interfaceId ||
            interfaceId == type(IMetadataV1).interfaceId ||
            interfaceId == type(IPayV1).interfaceId ||
            interfaceId == type(ICardsV1).interfaceId ||
            interfaceId == type(ICountV1).interfaceId ||
            interfaceId == type(IMintV1).interfaceId ||
            super.supportsInterface(interfaceId);
    }

    function interfaces() public pure returns(bytes4) {
        return type(ICardsV1).interfaceId;
    }

    // ------------------------ IBurnV1 functionality ----------------------------------

    // returns the cost of burning.
    function payBurnPrice() external view returns(uint256 burnWei) {
        return ipayPrice(BURN_INDEX);
    }
    // if the caller passes in zero, we'll validate that this token is soulbound and fetch
    // the token Id that needs to be burnt. 
    function payBurn(uint256 _tokenId) external payable {

        require(ipayValidateAmount(BURN_INDEX), "Wrong amount sent"); // must send the correct amount to burn.
        if(  0 == _tokenId) {
            require(IsSoulBound(),"must be soulBound");
            _tokenId = theTokenId(msg.sender);
        }
        _requireOwned(_tokenId);
        require(ownerOf(_tokenId)==msg.sender,"Not owner"); // only holder can burn the token

        // keep the series & spot in sync.
        uint256 seriesNum = m_toks[_tokenId].s_series;
        uint256 spotNum = m_toks[_tokenId].s_spot;
        series storage seriesItem = m_cards[seriesNum];
        require(seriesItem.m_spots[spotNum].s_total>0,"out of bal");
        seriesItem.m_spots[spotNum].s_total--;

        // record the payment for The Titled
        titledPayment();
        _burn(_tokenId);
        m_totalBurnt++;
    }

    // ------------------------ sendLock functionality ----------------------------------

    function paySendLockPrice() external view returns(uint256 sendLockWei) {
        require(IsSendLock()==true,"no sendlock");
        return ipayPrice(SENDLOCK_INDEX);
    }

    // There is a cost for using sendlock functionality.
    function sendLockRegister(address to, uint256 tokenId) public payable {
        _requireOwned(tokenId);
        require(IsSendLock()==true,"no sendlock");
        require(IsSoulBound()==false,"soulbound");
        require(_ownerOf(tokenId) == msg.sender, "Must own token");
        require(to != msg.sender, "Don't send to self");
        require(address(0) != to, "must provide valid destination");
        require(ipayPrice(SENDLOCK_INDEX) == msg.value, "Wrong amount sent");

        // record the payment for The Titled
        titledPayment();
        _sendLockRegister(to,tokenId);
    }
}
